<?php

namespace App\Listeners;

use App\Contact;
use App\Notifications\SystemActivityNotification;
use App\Transaction;
use App\User;
use Illuminate\Notifications\DatabaseNotification;

class SendInAppActivityNotification
{
    public function handle($event): void
    {
        $subject = $event->transaction ?? ($event->purchase ?? null);
        if (!$subject instanceof Transaction) {
            // If events carry different property names, try to resolve gracefully
            if (property_exists($event, 'transaction')) {
                $subject = $event->transaction;
            } elseif (property_exists($event, 'purchase')) {
                $subject = $event->purchase;
            }
        }

        $business_id = null;
        $type = null;
        $action = $event->action ?? null;
        $title = '';
        $body = '';
        $url = '';

        if ($subject instanceof Transaction) {
            $business_id = (int) $subject->business_id;
            $type = $subject->type; // 'sell' | 'purchase' | 'sell_transfer' | 'purchase_transfer'
            $action = $action ?? ($subject->wasRecentlyCreated ? 'created' : 'updated');
            if ($type === 'purchase') {
                $title = 'Purchase ' . ucfirst($action);
                $body  = 'Ref ' . ($subject->ref_no ?? '') . ' was ' . $action . '.';
                $url   = action([\App\Http\Controllers\PurchaseController::class, 'show'], [$subject->id]);
            } elseif ($type === 'sell_transfer' || $type === 'purchase_transfer') {
                $title = 'Intercompany Sale ' . ucfirst($action);
                $body  = 'Ref ' . ($subject->ref_no ?? '') . ' was ' . $action . '.';
                $url   = action([\App\Http\Controllers\StockTransferController::class, 'show'], [$subject->type === 'sell_transfer' ? $subject->id : ($subject->transfer_parent_id ?? $subject->id)]);
            } else {
                $title = 'Sale ' . ucfirst($action);
                $body  = 'Invoice ' . ($subject->invoice_no ?? '') . ' was ' . $action . '.';
                $url   = action([\App\Http\Controllers\SellController::class, 'show'], [$subject->id]);
            }
        } elseif (property_exists($event, 'transactionPayment') && $event->transactionPayment) {
            $payment = $event->transactionPayment;
            // Load linked transaction (avoid missing relation in some cases)
            $tx = $payment->transaction ?? Transaction::find($payment->transaction_id);
            if ($tx) {
                $subject = $tx;
                $business_id = (int) $tx->business_id;
                $type = $tx->type; // 'sell' | 'purchase'
                $action = $action ?? ($event instanceof \App\Events\TransactionPaymentDeleted ? 'payment_deleted' : ($event instanceof \App\Events\TransactionPaymentUpdated ? 'payment_updated' : 'payment_added'));
                if ($type === 'purchase') {
                    $title = 'Purchase ' . str_replace('_', ' ', ucfirst($action));
                    $body  = 'Ref ' . ($tx->ref_no ?? '') . ': payment was ' . str_replace('_', ' ', substr($action, 8)) . '.';
                    $url   = action([\App\Http\Controllers\PurchaseController::class, 'show'], [$tx->id]);
                } else {
                    $title = 'Sale ' . str_replace('_', ' ', ucfirst($action));
                    $body  = 'Invoice ' . ($tx->invoice_no ?? '') . ': payment was ' . str_replace('_', ' ', substr($action, 8)) . '.';
                    $url   = action([\App\Http\Controllers\SellController::class, 'show'], [$tx->id]);
                }
            }
        } elseif (property_exists($event, 'stock') && $event->stock) {
            $tx = $event->stock; // Transaction of type sell_transfer
            if ($tx) {
                $subject = $tx;
                $business_id = (int) $tx->business_id;
                $type = $tx->type; // sell_transfer
                $action = $action ?? ($event->action ?? ($tx->wasRecentlyCreated ? 'created' : 'updated'));
                $title = 'Intercompany Sale ' . ucfirst($action);
                $body  = 'Ref ' . ($tx->ref_no ?? '') . ' was ' . $action . '.';
                $url   = action([\App\Http\Controllers\StockTransferController::class, 'show'], [$tx->id]);
            }
        } else {
            return; // nothing recognizable
        }

        $payload = [
            'title' => $title,
            'body'  => $body,
            'url'   => $url,
            'meta'  => [
                'transaction_id' => $subject->id,
                'type' => $type,
                'action' => $action,
            ],
        ];

        // Build recipient list:
        // - Admins (superadmin) see all
        // - Actor (who did the action) sees their own
        // - Original creator (Added By) gets updates when others modify their record
        // - Commission Agent on the transaction always notified for create/update/payment changes
        $recipientIds = [];

        // Admins
        $businessUsers = User::where('business_id', $business_id)->get();
        foreach ($businessUsers as $user) {
            if ($user->can('superadmin')) {
                $recipientIds[$user->id] = true;
            }
        }

        // Ensure Admin role also gets sale payment events
        if (!empty($action) && strpos($action, 'payment_') === 0 && ($type === 'sell')) {
            foreach ($businessUsers as $user) {
                if (method_exists($user, 'hasRole') && $user->hasRole('Admin')) {
                    $recipientIds[$user->id] = true;
                }
            }
        }

        // Actor
        $actorId = auth()->id();
        if ($actorId) {
            $recipientIds[$actorId] = true;
        }

        // Original creator on updates by others
        if ($subject instanceof Transaction && !empty($subject->created_by)) {
            $isUpdateAction = !in_array($action, ['created', 'payment_added']) && $action !== null; // treat all non-created/new as updates
            if ($isUpdateAction && (int) $subject->created_by !== (int) $actorId) {
                $recipientIds[(int) $subject->created_by] = true;
            }
        }

        // Commission agent on the sale (can be int id or CSV of ids)
        if ($subject instanceof Transaction && !empty($subject->commission_agent)) {
            $ca = $subject->commission_agent;
            $ids = [];
            if (is_numeric($ca)) {
                $ids[] = (int) $ca;
            } elseif (is_array($ca)) {
                $ids = array_map('intval', $ca);
            } elseif (is_string($ca)) {
                foreach (preg_split('/[,;\s]+/', $ca) as $piece) {
                    if (is_numeric($piece)) $ids[] = (int) $piece;
                }
            }
            foreach ($ids as $cid) {
                if ($cid > 0) $recipientIds[$cid] = true;
            }
        }

        // Suppliers mapped to this purchase's contact
        if ($subject instanceof Transaction && $type === 'purchase' && !empty($subject->contact_id)) {
            $supplierUsers = User::where('business_id', $business_id)
                ->where('supplier_contact_id', (int) $subject->contact_id)
                ->get();
            foreach ($supplierUsers as $su) {
                $recipientIds[(int) $su->id] = true;
            }
        }

        // Notify unique recipients only (with de-duplication by recent record)
        foreach (array_keys($recipientIds) as $uid) {
            $u = $businessUsers->firstWhere('id', (int) $uid) ?? User::find((int) $uid);
            if ($u && !$this->isDuplicate($u->id, $payload)) {
                $u->notify(new SystemActivityNotification($payload));
            }
        }
    }

    private function isDuplicate(int $userId, array $payload): bool
    {
        $since = now()->subMinutes(3);
        return DatabaseNotification::where('notifiable_id', $userId)
            ->where('type', SystemActivityNotification::class)
            ->where('created_at', '>=', $since)
            ->where(function ($q) use ($payload) {
                $q->where('data->meta->transaction_id', $payload['meta']['transaction_id'] ?? null)
                  ->where('data->meta->action', $payload['meta']['action'] ?? null);
            })
            ->exists();
    }
}
